

# This file was *autogenerated* from the file echelon.sage
from sage.all_cmdline import *   # import sage library

_sage_const_0 = Integer(0); _sage_const_1 = Integer(1); _sage_const_2 = Integer(2); _sage_const_3 = Integer(3); _sage_const_100 = Integer(100); _sage_const_10 = Integer(10); _sage_const_5 = Integer(5)
def echelon(X):
  r'''
  Return X reduced to row-echelon form.

  INPUT:

  "X" - a matrix
  '''
  return copy(X.rref())

def echelon_positions(R):
  r'''
  If R is in reduced row-echelon form,
  return the list c such that
  len(c) is the rank of R and, for each j in {0,1,...,r-1},
  c[j] is the position of the leading 1 in row j of R.

  If R is not in reduced row-echelon form,
  raise an exception.
  
  INPUT:

  "R" - a matrix over a field
  '''

  r = R.rank()
  c = [min(b for b,Rab in enumerate(R[a]) if Rab != _sage_const_0 ) for a in range(r)]

  # "c_0 < c_1 < ... < c_{r-1}"
  assert c == sorted(set(c))

  for a in range(r):
    # "row 0 of R begins with a 1 in column c_0"
    # "row 1 of R begins with a 1 in column c_1"
    # etc.
    assert R[a][:c[a]] == _sage_const_0 
    assert R[a,c[a]] == _sage_const_1 

  for a in range(r):
    for b in range(R.nrows()):
      if b != a:
        # "this is the only nonzero entry in column c_0"
        # "the only nonzero entry in column c_1"
        # etc.
        assert R[b,c[a]] == _sage_const_0 

  for a in range(r,R.nrows()):
    # "all subsequent rows of R are 0"
    assert R[a] == _sage_const_0 

  return c

def is_systematic(R):
  r'''
  Return True if R is in systematic form,
  else False.

  INPUT:

  "R" - a matrix in reduced row-echelon form over a field
  '''
  c = echelon_positions(R)
  r = len(c)

  # "R has exactly zero rows, i.e., there are no zero rows"
  if R.nrows() != r: return False

  # "c_i = i for 0 <= i < r"
  okc = all(c[i] == i for i in range(r))

  # "equivalent to simply saying c_{r-1} == r-1,
  #  except in the degenerate case r = 0"
  if r > _sage_const_0 : assert okc == (c[r-_sage_const_1 ] == r-_sage_const_1 )

  if not okc: return False

  # "in other words, R has the form (I_r | T),
  #  where I is an rxr identity matrix."
  for a in range(r):
    for b in range(r):
      assert R[a,b] == (a == b)

  return True

def is_semi_systematic(R,mu,nu):
  r'''
  Return True if R is in systematic form,
  else False.

  INPUT:

  "R" - a matrix in reduced row-echelon form over a field

  "mu" - a nonnegative integer

  "nu" - an integer with nu >= mu
  '''
  c = echelon_positions(R)
  r = len(c)
  assert mu >= _sage_const_0 
  assert nu >= mu

  # "r >= mu"
  if r < mu: return False

  # "there are at least r-mu+nu columns"
  if R.ncols() < r-mu+nu: return False

  # "R has r rows"
  if R.nrows() != r: return False

  # "c_i = i for 0 <= i < r-mu"
  # c_i <= i-mu+nu for 0 <= i < r"
  okc = all(c[i] == i for i in range(r-mu)) and all(c[i] <= i-mu+nu for i in range(r))

  # "equivalent to simply c_{r-mu-1} = r-mu-1 and c_{r-1} <= r-mu+nu-1
  #  except in the degenerate case r = mu"
  if r > mu: assert okc == ((c[r-mu-_sage_const_1 ] == r-mu-_sage_const_1 ) and (c[r-_sage_const_1 ] <= r-mu+nu-_sage_const_1 ))

  return okc

# ----- miscellaneous tests

def test_manual():
  print('echelon misc')
  sys.stdout.flush()

  X = matrix(GF(_sage_const_2 ),[[_sage_const_1 ,_sage_const_1 ,_sage_const_0 ],[_sage_const_0 ,_sage_const_0 ,_sage_const_1 ]])
  assert X == echelon(X)
  assert echelon_positions(X) == [_sage_const_0 ,_sage_const_2 ]
  assert echelon(matrix(GF(_sage_const_2 ),[[_sage_const_0 ,_sage_const_0 ,_sage_const_1 ],[_sage_const_1 ,_sage_const_1 ,_sage_const_1 ]])) == X
  assert echelon(matrix(GF(_sage_const_2 ),[[_sage_const_1 ,_sage_const_1 ,_sage_const_1 ],[_sage_const_1 ,_sage_const_1 ,_sage_const_0 ]])) == X

  X = matrix(GF(_sage_const_2 ),[[_sage_const_1 ,_sage_const_0 ,_sage_const_1 ],[_sage_const_0 ,_sage_const_1 ,_sage_const_1 ]])
  assert echelon(X) == X
  assert echelon_positions(X) == [_sage_const_0 ,_sage_const_1 ]
  assert is_systematic(X)

  X = matrix(GF(_sage_const_2 ),[[_sage_const_1 ,_sage_const_1 ,_sage_const_0 ,_sage_const_1 ],[_sage_const_0 ,_sage_const_0 ,_sage_const_1 ,_sage_const_1 ]])
  assert echelon(X) == X
  assert echelon_positions(X) == [_sage_const_0 ,_sage_const_2 ]
  assert not is_semi_systematic(X,_sage_const_0 ,_sage_const_0 )
  assert not is_semi_systematic(X,_sage_const_1 ,_sage_const_1 )
  assert not is_semi_systematic(X,_sage_const_0 ,_sage_const_1 )
  assert is_semi_systematic(X,_sage_const_1 ,_sage_const_2 )
  assert not is_semi_systematic(X,_sage_const_0 ,_sage_const_2 )
  assert is_semi_systematic(X,_sage_const_1 ,_sage_const_3 )

def test_smallrandom():
  for q in range(_sage_const_100 ):
    q = ZZ(q)
    if not q.is_prime(): continue
    k = GF(q)

    nummatrices = _sage_const_0 
    numsystematic = _sage_const_0 
    numsemisystematic = _sage_const_0 
    for rows in range(_sage_const_10 ):
      for cols in range(_sage_const_10 ):
        nummatrices += _sage_const_1 
        X = matrix(k,[[randrange(q) for b in range(cols)] for a in range(rows)])
        R = echelon(X)
        assert R.row_space() == X.row_space()
        assert R.rank() == X.rank()
        c = echelon_positions(R)
        assert len(c) == R.rank()
        if len(c) == R.nrows() and (len(c) == _sage_const_0  or c[-_sage_const_1 ] == len(c)-_sage_const_1 ):
          numsystematic += _sage_const_1 
          assert is_systematic(R)
          assert matrix([Ra[:len(c)] for Ra in R]) == identity_matrix(len(c))
        else:
          assert not is_systematic(R)
          assert matrix([Ra[:len(c)] for Ra in R]) != identity_matrix(len(c))
        if len(c) == R.nrows():
          if len(c) == _sage_const_0 :
            assert is_semi_systematic(R,_sage_const_0 ,_sage_const_0 )
          else:
            assert is_semi_systematic(R,len(c),c[-_sage_const_1 ]+_sage_const_1 )
    print('echelon field %d nummatrices %d numsystematic %d' % (q,nummatrices,numsystematic))
    sys.stdout.flush()

    for mu in range(_sage_const_5 ):
      for nu in range(mu,mu+_sage_const_5 ):
        for rows in range(mu,mu+_sage_const_5 ):
          if rows == _sage_const_0 : continue
          for cols in range(rows-mu+nu,rows-mu+nu+_sage_const_5 ):
            X = matrix(k,[[a==b if b<rows else randrange(q) for b in range(cols)] for a in range(rows)])
            perm = list(range(nu))
            shuffle(perm)
            perm = [b for b in range(rows-mu)]+[rows-mu+b for b in perm]+[b for b in range(rows-mu+nu,cols)]
            assert len(perm) == cols
            X = X.matrix_from_columns(perm)
            assert is_semi_systematic(echelon(X),mu,nu)

if __name__ == '__main__':
  test_manual()
  test_smallrandom()

