use std::{
    sync::{Arc, Mutex},
    time::Instant,
};

use smithay::{
    backend::renderer::utils::on_commit_buffer_handler,
    delegate_compositor, delegate_foreign_toplevel_list, delegate_output, delegate_seat,
    delegate_shm, delegate_xdg_shell,
    input::{pointer::CursorImageStatus, Seat, SeatHandler, SeatState},
    output::Output,
    utils::{Serial, Size},
    wayland::{
        buffer::BufferHandler,
        compositor::{
            with_states, with_surface_tree_downward, CompositorClientState, CompositorHandler,
            CompositorState, SurfaceAttributes, TraversalAction,
        },
        foreign_toplevel_list::{ForeignToplevelListHandler, ForeignToplevelListState},
        output::OutputHandler,
        shell::xdg::{
            PopupSurface, PositionerState, ToplevelSurface, XdgShellHandler, XdgShellState,
        },
        shm::{ShmHandler, ShmState},
    },
};
use wayland_protocols::{
    ext::{
        image_capture_source::v1::server::ext_foreign_toplevel_image_capture_source_manager_v1::ExtForeignToplevelImageCaptureSourceManagerV1,
        image_copy_capture::v1::server::ext_image_copy_capture_manager_v1::ExtImageCopyCaptureManagerV1,
    },
    xdg::shell::server::xdg_toplevel,
};
use wayland_server::{
    protocol::{wl_buffer::WlBuffer, wl_seat::WlSeat, wl_surface::WlSurface},
    Client, DisplayHandle,
};

use crate::{ext_image_copy_capture::SessionHandle, utils::WlSurfaceExt};

pub struct App {
    pub(crate) compositor_state: CompositorState,
    pub(crate) xdg_shell_state: XdgShellState,
    pub(crate) shm_state: ShmState,
    pub(crate) seat_state: SeatState<Self>,
    pub(crate) foreign_toplevel_list_state: ForeignToplevelListState,
    start_time: Instant,
}

impl App {
    pub fn new(dh: &DisplayHandle) -> Self {
        let compositor_state = CompositorState::new::<App>(dh);
        let xdg_shell_state = XdgShellState::new::<App>(dh);
        let foreign_toplevel_list_state = ForeignToplevelListState::new::<App>(dh);
        let shm_state = ShmState::new::<App>(dh, vec![]);
        dh.create_global::<App, ExtForeignToplevelImageCaptureSourceManagerV1, _>(1, ());
        dh.create_global::<App, ExtImageCopyCaptureManagerV1, _>(1, ());
        let output = Output::new(
            "fönster".to_owned(),
            smithay::output::PhysicalProperties {
                size: Size::new(1024, 1024),
                subpixel: smithay::output::Subpixel::Unknown,
                make: "fluxfingers".to_owned(),
                model: "fönster".to_owned(),
            },
        );
        output.create_global::<App>(dh);
        let seat_state = SeatState::new();

        App {
            compositor_state,
            xdg_shell_state,
            foreign_toplevel_list_state,
            shm_state,
            seat_state,
            start_time: Instant::now(),
        }
    }

    pub fn send_frame_callbacks(&self) {
        let time = self.start_time.elapsed().as_millis() as u32;
        for surface in self.xdg_shell_state.toplevel_surfaces() {
            let surface = surface.wl_surface();
            with_surface_tree_downward(
                surface,
                (),
                |_, _, &()| TraversalAction::DoChildren(()),
                |_surf, states, &()| {
                    for callback in states
                        .cached_state
                        .get::<SurfaceAttributes>()
                        .current()
                        .frame_callbacks
                        .drain(..)
                    {
                        callback.done(time);
                    }
                },
                |_, _, &()| true,
            );
        }
    }
}

impl CompositorHandler for App {
    fn compositor_state(&mut self) -> &mut CompositorState {
        &mut self.compositor_state
    }

    fn client_compositor_state<'a>(&self, client: &'a Client) -> &'a CompositorClientState {
        &client.get_data::<crate::ClientState>().unwrap().0
    }

    fn commit(&mut self, surface: &WlSurface) {
        with_states(surface, |states| {
            let Some(sessions) = states.data_map.get::<Arc<Mutex<Vec<SessionHandle>>>>() else {
                return;
            };
            let mut guard = states.cached_state.get::<SurfaceAttributes>();
            if let Some(smithay::wayland::compositor::BufferAssignment::NewBuffer(buf)) =
                &guard.current().buffer
            {
                sessions
                    .lock()
                    .unwrap()
                    .iter()
                    .for_each(|session| session.update_buffer(buf.clone()));
            }
        });
        // for good measure
        on_commit_buffer_handler::<Self>(surface);
    }
}

impl XdgShellHandler for App {
    fn xdg_shell_state(&mut self) -> &mut XdgShellState {
        &mut self.xdg_shell_state
    }

    fn new_toplevel(&mut self, surface: ToplevelSurface) {
        with_states(surface.wl_surface(), |states| {
            let handle = states
                .data_map
                .get_or_insert(|| self.foreign_toplevel_list_state.new_toplevel::<App>("", ""));
            handle.user_data().get_or_insert(|| surface.clone());
        });
        surface.with_pending_state(|state| {
            state.states.set(xdg_toplevel::State::Activated);
            state.size = Some(Size::new(128,128));
            state.decoration_mode = Some(wayland_protocols::xdg::decoration::zv1::server::zxdg_toplevel_decoration_v1::Mode::ServerSide);
        });
        surface.send_configure();
    }

    fn toplevel_destroyed(&mut self, surface: ToplevelSurface) {
        if let Some(handle) = surface.wl_surface().get_handle() {
            self.foreign_toplevel_list_state.remove_toplevel(&handle);
        }
    }

    fn new_popup(&mut self, _surface: PopupSurface, _positioner: PositionerState) {}

    fn grab(&mut self, _surface: PopupSurface, _seat: WlSeat, _serial: Serial) {}

    fn title_changed(&mut self, surface: ToplevelSurface) {
        let Some(title) = surface.wl_surface().get_title() else {
            return;
        };
        let Some(handle) = surface.wl_surface().get_handle() else {
            return;
        };

        handle.send_title(&title);
        handle.send_done();
    }

    fn app_id_changed(&mut self, surface: ToplevelSurface) {
        let Some(app_id) = surface.wl_surface().get_app_id() else {
            return;
        };
        let Some(handle) = surface.wl_surface().get_handle() else {
            return;
        };

        handle.send_app_id(&app_id);
        handle.send_done();
    }

    fn reposition_request(
        &mut self,
        _surface: PopupSurface,
        _positioner: PositionerState,
        _token: u32,
    ) {
    }
}

impl ForeignToplevelListHandler for App {
    fn foreign_toplevel_list_state(&mut self) -> &mut ForeignToplevelListState {
        &mut self.foreign_toplevel_list_state
    }
}

impl ShmHandler for App {
    fn shm_state(&self) -> &ShmState {
        &self.shm_state
    }
}

impl BufferHandler for App {
    fn buffer_destroyed(&mut self, _buffer: &WlBuffer) {}
}

impl OutputHandler for App {}

impl SeatHandler for App {
    type KeyboardFocus = WlSurface;
    type PointerFocus = WlSurface;
    type TouchFocus = WlSurface;

    fn seat_state(&mut self) -> &mut SeatState<Self> {
        &mut self.seat_state
    }

    fn focus_changed(&mut self, _seat: &Seat<Self>, _focused: Option<&WlSurface>) {}
    fn cursor_image(&mut self, _seat: &Seat<Self>, _image: CursorImageStatus) {}
}

delegate_xdg_shell!(App);
delegate_compositor!(App);
delegate_shm!(App);
delegate_seat!(App);
delegate_output!(App);
delegate_foreign_toplevel_list!(App);
